<?php
/**
 * core.framework
 *
 * @category   Core
 * @package    Core_Storage_Strategy_Abstract
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2011. Burza d.o.o. (http://web.burza.hr/en/)
 * @license    proprietary
 */

/**
 * @category   Core
 * @package    Core_Storage_Strategy_Abstract
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2011. Burza d.o.o. (http://web.burza.hr/en/)
 * @license    proprietary
 * @group      Core_Storage_Strategy_Abstract
 */
class Core_Storage_Strategy_AbstractTest extends PHPUnit_Framework_TestCase
{
    /**
     * @var Core_Storage_Strategy_Abstract_Strategy_Abstract
     */
    protected $_object;

    protected function setUp()
    {
        $this->_object = $this->getMockForAbstractClass('Core_Storage_Strategy_Abstract');
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setOptions
     */
    public function testCanSetOptions()
    {
        $path    = '/one/two/3';
        $options = array('root' => $path);
        $this->_object->setOptions($options);
        $this->assertEquals($path, $this->_object->getRoot());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setDomain
     * @covers Core_Storage_Strategy_Abstract::getDomain
     */
    public function testCanSetAndGetDomain()
    {
        $this->_object->setDomain('https://www.coolinarika.com');
        $this->assertEquals('https://www.coolinarika.com', $this->_object->getDomain());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setDomain
     * @covers Core_Storage_Strategy_Abstract::getDomain
     */
    public function testCanSetAndGetHttpsDomain()
    {
        $domain = 'https://cdn.rtl.hr';
        $this->_object->setDomain($domain);
        $this->assertEquals($domain, $this->_object->getDomain());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setDomain
     * @covers Core_Storage_Strategy_Abstract::getDomain
     */
    public function testCanSetAndGetHttpDomain()
    {
        $domain = 'http://123.dot.com';
        $this->_object->setDomain($domain);
        $this->assertEquals($domain, $this->_object->getDomain());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::getDomain
     * @expectedException InvalidArgumentException
     */
    public function testGetDomainThrowsInvalidArgumentExceptionWhenDomainNotSet()
    {
        $this->setExpectedException('InvalidArgumentException');
        $this->_object->getDomain();
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setDomain
     * @expectedException InvalidArgumentException
     */
    public function testThrowsInvalidArgumentExceptionWhenSettingInvalidDomain()
    {
        $this->setExpectedException('InvalidArgumentException');
        $this->_object->setDomain('www.coolinarika.com');
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setConfig
     */
    public function testCanSetConfig()
    {
        $path   = '/one/two/3';
        $config = $this->getMock('Zend_Config', null, array(array('root' => $path)));
        $this->_object->setConfig($config);
        $this->assertEquals($path, $this->_object->getRoot());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::__construct
     */
    public function testCanSetOptionsViaConstructor()
    {
        $path    = '/one/two/3';
        $options = array('root' => $path);
        $object  = $this->getMockForAbstractClass('Core_Storage_Strategy_Abstract', array($options));
        $this->assertEquals($path, $object->getRoot());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::__construct
     */
    public function testCanSetConfigViaConstructor()
    {
        $path   = '/one/two/3';
        $config = $this->getMock('Zend_Config', null, array(array('root' => $path)));
        $object  = $this->getMockForAbstractClass('Core_Storage_Strategy_Abstract', array($config));
        $this->assertEquals($path, $object->getRoot());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setOptions
     */
    public function testPassingAnUnknownOptionsThrowsAnInvalidArgumentException()
    {
        $this->setExpectedException('InvalidArgumentException');
        $this->_object->setOptions(array('no-such' => 'option'));
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setOptions
     */
    public function testWillSkipFrobiddenOptions()
    {
        $this->_object->setRoot('/foo');
        $options = array('options' => array('root' => 'ignored'));
        $this->_object->setOptions($options);

        $this->assertNotEquals('ignored', $this->_object->getRoot());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setRoot
     * @covers Core_Storage_Strategy_Abstract::getRoot
     */
    public function testCanSetAndGetRoot()
    {
        $this->_object->setRoot('/web/root');

        $this->assertEquals('/web/root', $this->_object->getRoot());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::getRoot
     */
    public function testGetRootThrowsInvalidArgumentExceptionWhenRootNotSet()
    {
        $this->setExpectedException('InvalidArgumentException');

        $this->_object->getRoot();
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setUrlRoot
     * @covers Core_Storage_Strategy_Abstract::getUrlRoot
     */
    public function testCanSetAndGetUrlRoot()
    {
        $this->_object->setUrlRoot('/url/root');

        $this->assertEquals('/url/root', $this->_object->getUrlRoot());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::getUrlRoot
     */
    public function testCallingGetUrlRootWithoutBeingSetThrowsAnInvalidArgumentException()
    {
        $this->setExpectedException('InvalidArgumentException');

        $this->_object->getUrlRoot();
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setFileMode
     * @covers Core_Storage_Strategy_Abstract::getFileMode
     */
    public function testCanSetAndGetFileMode()
    {
        $this->_object->setFileMode(0123);

        $this->assertEquals(0123, $this->_object->getFileMode());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::getFileMode
     */
    public function testDefaultFileModeIs0644()
    {
        $this->assertEquals(0644, $this->_object->getFileMode());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setDirectoryMode
     * @covers Core_Storage_Strategy_Abstract::getDirectoryMode
     */
    public function testCanSetAndGetDirectoryMode()
    {
        $this->_object->setDirectoryMode(0123);

        $this->assertEquals(0123, $this->_object->getDirectoryMode());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setRoot
     */
    public function testSettingRootPathRemovesTrailingSlashes()
    {
        $this->_object->setRoot('/some/path/');
        $this->assertEquals('/some/path', $this->_object->getRoot());
    }

    /**
     * @covers Core_Storage_Strategy_Abstract::setUrlRoot
     */
    public function testSettingUrlRootPathRemovesTrailingSlash()
    {
        $this->_object->setUrlRoot('http://some/path/');
        $this->assertEquals('http://some/path', $this->_object->getUrlRoot());
    }
}
