<?php
/**
 * core.framework
 *
 * @category   Core
 * @package    Core_Security
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2011. Burza d.o.o. (http://web.burza.hr/en/)
 * @license    proprietary
 */

class Core_SecurityRolesTest extends PHPUnit_Framework_TestCase {

    /**
     * @var Core_Security
     */
    protected $_object;

    protected function setUp()
    {
        $options = array(
            'roles'     => array(
                array(
                    'name' => 'user'
                ),
                array(
                    'name' => 'owner'
                )
            ),
            'resources' => array(
                'moduleNoMethodPrivileges' => array(
                    'privileges' => array(
                        'allow' => array(
                            'user' => 'delete'
                        )
                    )
                ),
                'moduleOnlyUserAllowed' => array(
                    'privileges' => array(
                        'allow' => array(
                            'user' => 'delete'
                        )
                    )
                ),
                'moduleOnlyOwnerAllowed' => array(
                    'privileges' => array(
                        'allow' => array(
                            'owner' => 'update'
                        )
                    )
                ),
                'moduleBothAllowed' => array(
                    'privileges' => array(
                        'allow' => array(
                            'owner' => 'update',
                            'user'  => '*'
                        )
                    )
                ),
                'moduleMethodOwnerAllowedUserDenied' => array(
                    'privileges' => array(
                        'allow' => array(
                            'owner' => '*',
                            'user'  => 'update'
                        ),
                        'deny' => array(
                            'user' => 'update'
                        )
                    )
                ),
                'moduleMethodOwnerAllowedUserDeniedAll' => array(
                    'privileges' => array(
                        'allow' => array(
                            'owner' => 'update',
                            'user'  => 'update'
                        ),
                        'deny' => array(
                            'user' => '*'
                        )
                    )
                )
            ),
        );

        $config = $this->getMock('Zend_Config', null, array($options));
        $this->_object = new Core_Security($config);
    }

    public function getSampleRoles()
    {
        return array(
            array(  'user',                   'moduleNoMethodPrivileges',                 'update', false   ),
            array(  'owner',                  'moduleNoMethodPrivileges',                 'update', false   ),

            array(  'user',                   'moduleOnlyUserAllowed',                    'delete', true    ),
            array(  array('owner', 'user'),   'moduleOnlyUserAllowed',                    'delete', true    ),
            array(  'user',                   'moduleOnlyUserAllowed',                    'update', false   ),
            array(  'owner',                  'moduleOnlyUserAllowed',                    'delete', false   ),
            array(  array('owner', 'user'),   'moduleOnlyUserAllowed',                    'delete', true    ),
            array(  'owner',                  'moduleOnlyOwnerAllowed',                   'update', true    ),
            array(  'user',                   'moduleOnlyOwnerAllowed',                   'update', false   ),
            array(  array('owner', 'user'),   'moduleOnlyOwnerAllowed',                   'update', true    ),

            array(  'user',                   'moduleBothAllowed',                        'update', true    ),
            array(  'owner',                  'moduleBothAllowed',                        'update', true    ),

            array(  'owner',                  'moduleMethodOwnerAllowedUserDenied',       'update', true    ),
            array(  'user',                   'moduleMethodOwnerAllowedUserDenied',       'update', false   ),
            array(  'owner',                  'moduleMethodOwnerAllowedUserDenied',       'update', true    ),
            array(  array('owner', 'user'),   'moduleMethodOwnerAllowedUserDenied',       'update', false   ),

            array(  array('owner', 'user'),   'moduleMethodOwnerAllowedUserDeniedAll',    'update', false   ),
            array(  array('user', 'owner'),   'moduleMethodOwnerAllowedUserDeniedAll',    'update', false   ),
        );
    }

    /**
     * @dataProvider getSampleRoles
     *
     * @param string|array $roles
     * @param string $method
     * @param string $privilege
     * @param bool $shouldPass
     */
    public function testSecurityRoles($roles, $method, $privilege, $shouldPass)
    {
        if ($shouldPass) {
            $this->assertTrue($this->_object->isAllowed($roles, $method, $privilege));
        } else {
            $this->assertFalse($this->_object->isAllowed($roles, $method, $privilege));
        }
    }

} 