<?php
/**
 * core.framework
 *
 * @category   Core
 * @package    Core_Calendar
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2011. Burza d.o.o. (http://web.burza.hr/en/)
 * @license    proprietary
 */

/**
 * @category   Core
 * @package    Core_Calendar
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2011. Burza d.o.o. (http://web.burza.hr/en/)
 * @license    proprietary
 * @group      Core_Calendar
 */
class Core_CalendarTest extends PHPUnit_Framework_TestCase
{
    /**
     * @var Core_Calendar
     */
    protected $_object;

    protected function setUp()
    {
        $this->_object = new Core_Calendar;
    }

    /**
     * @covers Core_Calendar::setDateValidator
     * @covers Core_Calendar::getDateValidator
     */
    public function testCanSetDateValidator()
    {
        $validator = $this->getMock('Core_Validate_Date', null, array('date'));
        $this->_object->setDateValidator($validator);

        $this->assertSame($validator, $this->_object->getDateValidator());
    }

    /**
     * @covers Core_Calendar::getDateValidator
     */
    public function testIfNoDateValidatorSpecifiedWillCreateOne()
    {
        $this->assertInstanceOf('Core_Validate_Date', $this->_object->getDateValidator());
    }

    /**
     * @covers Core_Calendar::setDate
     * @covers Core_Calendar::getDate
     */
    public function testCansetDate()
    {
        $this->_addDateValidator(true);
        $date = array(
            'year'  => 2000,
            'month' => 12,
            'day'   => 31
        );
        $this->_object->setDate($date);

        $this->assertEquals($date, $this->_object->getDate());
    }

    /**
     * @covers Core_Calendar::setDate
     * @covers Core_Calendar::getDate
     * @covers Core_Calendar::<protected>
     */
    public function testCansetDateAsString()
    {
        $this->_addDateValidator(true);
        $date = array(
            'year'  => 2000,
            'month' => 12,
            'day'   => 31
        );
        $dateAsString = '2000-12-31';
        $this->_object->setDate($dateAsString);

        $this->assertEquals($date, $this->_object->getDate());
    }

    /**
     * @covers Core_Calendar::getDate
     */
    public function testIfDateNotSetWillDefaultToToday()
    {
        $this->_addDateValidator(true);

        $todayAsArray = array(
            'year'  => date('Y'),
            'month' => date('m'),
            'day'   => date('d'),
        );
        $this->assertEquals($todayAsArray, $this->_object->getDate());
    }

    /**
     * @covers Core_Calendar::setDate
     * @covers Core_Calendar::<protected>
     */
    public function testPassingAnInvalidFormatCurrentDateAsStringThrowsAnInvalidArgumentException()
    {
        $this->setExpectedException('InvalidArgumentException');

        $invalidDateString = '2011-1212-12';
        $this->_object->setDate($invalidDateString);
    }

    /**
     * @covers Core_Calendar::setDate
     * @covers Core_Calendar::<protected>
     */
    public function testPassingAnInvalidValueCurrentDateAsStringThrowsAnInvalidArgumentException()
    {
        $this->_addDateValidator(false);
        $this->setExpectedException('InvalidArgumentException');

        $invalidDateString = '2011-34-12';
        $this->_object->setDate($invalidDateString);
    }

    /**
     * @covers Core_Calendar::setDate
     */
    public function testPassingAnInvalidValueCurrentDateAsArrayThrowsAnInvalidArgumentException()
    {
        $this->_addDateValidator(false);
        $this->setExpectedException('InvalidArgumentException');

        $invalidDateArray = array(
            'year'  => 2010,
            'month' => 34,
            'day'   => 999,
        );
        $this->_object->setDate($invalidDateArray);
    }

    /**
     * @covers Core_Calendar::getCurrentMonth
     */
    public function testCanGetCurrentMonth()
    {
        $this->_addDateValidator(true);

        $todayAsArray = array(
            'year'  => date('Y'),
            'month' => date('m'),
            'day'   => date('d'),
        );
        $this->_object->setDate($todayAsArray);

        $currentMonth = $this->_object->getCurrentMonth();

        $this->assertInstanceOf('Core_Calendar_Interval_Month', $currentMonth);
    }

    protected function _addDateValidator($returnValue = true)
    {
        $validator = $this->getMock('Core_Validate_Date', array('isValid'), array('date'));
        $validator
            ->expects($this->once())
            ->method('isValid')
            ->will($this->returnValue($returnValue));

        $this->_object->setDateValidator($validator);
    }
}
