<?php
/**
 * core.framework
 *
 * @category   Core
 * @package    Core_Asset
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2013. Burza d.o.o. (http://web.burza.hr/en/)
 * @license    proprietary
 */

// sadly, vfsStream does not follow PSR-0, burn in flames
require_once 'vfsStream/vfsStream.php';

/**
 * @category   Core
 * @package    Core_Asset
 * @subpackage UnitTests
 * @copyright  Copyright (c) 2013. Burza d.o.o. (http://web.burza.hr/en/)
 * @license    proprietary
 * @group      Core_Asset
 */


class Core_AssetTest extends PHPUnit_Framework_TestCase
{
    /**
     * @var Core_Asset
     */
    protected $_object;

    /**
     * vfsStream mock for filesystem
     *
     * @var string
     */
    protected $_root        = 'vfs://assets';

    /**
     * Example CSS, we mock file with actual data quite a lot (DRY)
     */
    protected $_cssCode     = null;

    protected function setUp()
    {
        // using vfsStream for mocking out the filesystem!
        if (class_exists('vfsStream', false) === false) {
            $this->markTestSkipped('Core_AssetTest requires vfsStream, see http://vfs.bovigo.org/.');
        }

        $this->_cssCode = "
/** STRUCTURE **/
html        { overflow-y: scroll; }

/** GRID **/
/* container */
.container_full  { width: 100%; text-align: center; display: inline-block; position: relative; }
.container       { width: 984px; text-align: left; margin: 0 auto; zoom: 1; position: relative; display: inline-block; }
        ";

        vfsStream::setup('assets');
        mkdir($this->_root .'/assets',                  0777);
        mkdir($this->_root .'/some-css/',               0777);
        mkdir($this->_root .'/some-css/lib/',           0777);
        mkdir($this->_root .'/some-css/lib/vendor/',    0777);
    }


    /**
     * @covers Core_Asset::__toString
     */
    public function testGetAssetContentsAsString()
    {
        $filename   = $this->_root . '/asset1.css';
        file_put_contents($filename, $this->_cssCode);
        $asset      = new Core_Asset($filename);

        $this->assertEquals($this->_cssCode, (string) $asset);
    }

    /**
     * @covers Core_Asset::dump
     */
    public function testDumpReturnsAssetContents()
    {
        $this->markTestIncomplete('TODO');
        
        $filename   = $this->_root . '/asset1.css';
        file_put_contents($filename, $this->_cssCode);
        $asset      = new Core_Asset($filename);

        $this->assertEquals($this->_cssCode, $asset->dump());
    }

    /**
     * @covers Core_Asset::dump
     *
     * @expectedException InvalidArgumentException
     */
    public function testDumpThrowsInvalidArgumentExceptionWhenFileDoesntExist()
    {
        $this->markTestIncomplete('TODO');
        
        $filename   = $this->_root . '/asset1.css';
        $asset      = new Core_Asset($filename);

        $this->assertEquals($this->_cssCode, $asset->dump());
    }



    /**
     * @covers Core_Asset::setCacheBust
     * @covers Core_Asset::isCacheBust
     */
    public function testCanSetAndGetCacheBusting()
    {
        $filename   = $this->_root . '/asset1.css';
        file_put_contents($filename, $this->_cssCode);
        $asset      = new Core_Asset($filename);

        $asset->setCacheBust(true);
        $this->assertTrue($asset->isCacheBust());
    }

    /**
     * @covers Core_Asset::getUrl
     * @covers Core_Asset::setCacheBust
     * @covers Core_Asset::isCacheBust
     */
    public function testGetUrlReturnsValidCacheBusterStringWhenCacheBustSetToTrue()
    {
        $this->markTestIncomplete('TODO');
        
        $filename   = $this->_root . '/asset1.css';
        file_put_contents($filename, $this->_cssCode);
        $dataHash   = md5($this->_cssCode);

        $storage    = $this->getMock('Core_Storage_Strategy_Hash', array('getUrlRoot', 'getDomain'), array());
        $storage
            ->expects($this->any())
            ->method('getUrlRoot')
            ->will($this->returnValue($this->_root));
        $storage
            ->expects($this->any())
            ->method('getDomain')
            ->will($this->returnValue('http://www.coolinarika.com'));

        $asset      = new Core_Asset($filename, null, null, null, '/assets/asset1.css');
        $asset->setStorage($storage);
        $asset->setCacheBust(true);

        $expectedUrl    = 'http://www.coolinarika.com/assets/asset1.css?v='. $dataHash;

        $this->assertEquals($expectedUrl, $asset->getUrl());
    }

    /**
     * @covers Core_Asset::getUrl
     */
    public function testGetUrlReturnsValidUrl()
    {
        // storage mora imat definiran domain
        // storage mora imat definiran url

        $filename   = $this->_root . '/asset1.css';
        file_put_contents($filename, $this->_cssCode);
        $dataHash   = md5($this->_cssCode);

        $storage    = $this->getMock('Core_Storage_Strategy_Hash', array('getUrlRoot', 'getDomain'), array());
        $storage
            ->expects($this->any())
            ->method('getUrlRoot')
            ->will($this->returnValue($this->_root));
        $storage
            ->expects($this->any())
            ->method('getDomain')
            ->will($this->returnValue('http://www.coolinarika.com'));

        $asset = new Core_Asset($filename, null, null, null, '/assets/asset1.css');
        $asset->setStorage($storage);

        $expectedUrl    = 'http://www.coolinarika.com/assets/asset1.css';

        $this->assertEquals($expectedUrl, $asset->getUrl());
    }

    /**
     * @covers Core_Asset::getUrl
     */
    public function testGetUrlReturnsValidUrlWithCacheBust()
    {
        $filename   = $this->_root . '/asset1.css';
        file_put_contents($filename, $this->_cssCode);
        $dataHash   = md5($this->_cssCode);

        $storage    = $this->getMock('Core_Storage_Strategy_Hash', array('getUrlRoot', 'getDomain'), array());
        $storage
            ->expects($this->any())
            ->method('getUrlRoot')
            ->will($this->returnValue($this->_root));
        $storage
            ->expects($this->any())
            ->method('getDomain')
            ->will($this->returnValue('http://www.coolinarika.com'));

        $asset = new Core_Asset($filename, null, null, null, '/assets/asset1.css');
        $asset
            ->setStorage($storage)
            ->setCacheBust(true);

        $expectedUrl    = 'http://www.coolinarika.com/assets/asset1.css?v='. md5($this->_cssCode);

        $this->assertEquals($expectedUrl, $asset->getUrl());
    }



    /**
     * @covers Core_Asset::getUrlPath
     */
    public function testGetUrlPathReturnsValidUrlPath()
    {
        $filename   = $this->_root . '/some-css/lib/vendor/asset1.css';
        file_put_contents($filename, $this->_cssCode);
        $dataHash   = md5($this->_cssCode);

        $storage    = $this->getMock('Core_Storage_Strategy_Hash', array('getUrlRoot'), array());
        $storage
            ->expects($this->any())
            ->method('getUrlRoot')
            ->will($this->returnValue($this->_root));

        $asset = new Core_Asset($filename, null, null, null, '/some-css/lib/vendor/asset1.css');
        $asset->setStorage($storage);

        $expectedUrl    = '/some-css/lib/vendor/asset1.css';

        $this->assertEquals($expectedUrl, $asset->getUrlPath());
    }

    /**
     * @covers Core_Asset::getUrlPath
     */
    public function testGetUrlPathReturnsValidUrlPathWithCacheBust()
    {
        $filename   = $this->_root . '/some-css/lib/vendor/asset1.css';
        file_put_contents($filename, $this->_cssCode);
        $dataHash   = md5($this->_cssCode);

        $storage    = $this->getMock('Core_Storage_Strategy_Hash', array('getUrlRoot'), array());
        $storage
            ->expects($this->any())
            ->method('getUrlRoot')
            ->will($this->returnValue($this->_root));

        $asset = new Core_Asset($filename, null, null, null, '/some-css/lib/vendor/asset1.css');
        $asset
            ->setStorage($storage)
            ->setCacheBust(true);

        $expectedUrl    = '/some-css/lib/vendor/asset1.css?v='. md5($this->_cssCode);

        $this->assertEquals($expectedUrl, $asset->getUrlPath());
    }





}
