<?php
/**
 * core.framework
 *
 * @category  Core
 * @package   Core_File
 * @copyright Copyright (c) 2011. Burza d.o.o. (http://web.burza.hr/en/)
 * @license   proprietary
 */

/**
 * @category  Core
 * @package   Core_File
 * @copyright Copyright (c) 2011. Burza d.o.o. (http://web.burza.hr/en/)
 * @license   proprietary
 */
class Core_File
{
    /**
     * @var array Mapping of type MIME type => user-expected MIME type
     */
    static protected $_mimeMap = array(
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'application/vnd.microsoft.office.word',
        'application/msword' => 'application/vnd.microsoft.office.word',
        'application/vnd.openxmlformats-officedocument.presentationml.presentation' => 'application/vnd.microsoft.office.powerpoint',
        'application/vnd.ms-powerpoint' => 'application/vnd.microsoft.office.powerpoint',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'application/vnd.microsoft.office.excel',
        'application/vnd.ms-office' => 'application/vnd.microsoft.office.excel',
        'application/x-7z-compressed' => 'application/7z',
        'application/x-zip'           => 'application/zip',
        'application/ogg'             => 'audio/ogg',
        'image/x-ms-bmp'              => 'image/bmp',
        'video/x-ms-asf'              => 'video/amf',
        'video/x-flv'                 => 'video/flv',
    );

    /**
     * @var string
     */
    protected $_name;

    /**
     * @var string
     */
    protected $_path;

    /**
     * @var int
     */
    protected $_size;

    /**
     * @var string
     */
    protected $_type;

    /**
     * @var string
     */
    protected $_extension;
    
    /**
     * @param string $mimeType
     * 
     * @return string
     */
    public static function getMimeType($mimeType)
    {
        if (false !== ($idx = strpos($mimeType, ';')) || false !== ($idx = strpos($mimeType, ' '))) {
            // some old fileinfo instances omit the ;
            $type = substr($mimeType, 0, $idx);
        } else {
            $type = $mimeType;
        }
        return $type;
    }

    /**
     * @param string $path File path
     * @param string $name File name
     * @param int    $size File size
     * @param string $type File MIME type
     */
    public function __construct($path, $name = null, $size = null, $type = null)
    {
        $this
            ->setName($name)
            ->setPath($path)
            ->setSize($size)
            ->setType($type);
    }

    /**
     * @param string $path
     *
     * @return Core_File
     */
    public function setPath($path)
    {
        $this->_path = $path;
        return $this;
    }

    /**
     * @return string
     */
    public function getPath()
    {
        return $this->_path;
    }

    /**
     * @param string $name
     *
     * @return Core_File
     */
    public function setName($name)
    {
        $this->_name = $name;
        return $this;
    }

    /**
     * @return string
     */
    public function getName()
    {
        if (null === $this->_name) {
            $this->_name = basename($this->getPath());
        }
        return $this->_name;
    }

    /**
     * @param int $size
     *
     * @return Core_File
     */
    public function setSize($size)
    {
        $this->_size = $size;
        return $this;
    }

    /**
     * @return int
     */
    public function getSize()
    {
        if (null === $this->_size) {
            $this->_size = filesize($this->getPath());
        }
        return $this->_size;
    }

    /**
     * @param string $type File MIME type
     *
     * @return Core_File
     */
    public function setType($type)
    {
        $this->_type = $type;
        return $this;
    }

    /**
     * @param string $mime
     *
     * @return string File MIME type
     */
    public function getType($mime = null)
    {
        if (null === $this->_type) {
            if (null === $mime) {
                $mime = mime_content_type($this->getPath());                
            }
            $type = self::getMimeType($mime);

            if (array_key_exists($type, self::$_mimeMap)) {
                $type = self::$_mimeMap[$type];
            }
            $this->_type = $type;
        }
        return $this->_type;
    }

    /**
     * @param string $extension File extension
     *
     * @return Core_File
     */
    public function setExtension($extension)
    {
        $this->_extension = $extension;
        return $this;
    }

    /**
     * @return string File extension
     */
    public function getExtension()
    {
        if (null === $this->_extension) {
            $file             = pathinfo($this->getName());
            if (!isset($file['extension'])) {
                $file['extension'] = '';
            }
            $this->_extension = $file['extension'];
        }
        return $this->_extension;
    }
}
