<?php

/**
 * Core Detector Request
 *
 * Request object used to build an array of data that's being sent to SERVER API
 *
 * @author: Zwer 01/2013
 *
 * @copyright (c) 2013, Web.Burza
 *
 */

class Core_Detector_Request {

    /**
     * Server side information data
     */
    protected $_serverSideInfo      = array();

    /**
     * Client side information data
     */
    protected $_clientSideInfo      = array();

    /**
     * Custom data from both server and client side
     */
    protected $_customData          = array();

    /**
     * Environment information
     */
    protected $_environmentInfo      = array();

    /**
     * Detection properties to be returned upon successful request to server (API)
     */
    protected $_detectionProperties = array();


    /**
     * Constructor
     */
    public function __construct()
    {
        // server returns detection_id by default
        $this->_detectionProperties = array('detection_id');
    }

    /**
     * Format request data as JSON encoded string
     */
    public function toJson()
    {
        return json_encode($this->getData());

    }

    /**
     * Set Data for request
     * @param   array   $data
     */
    public function setData($data)
    {
        foreach ($data as $name => $value) {
            $method = 'set' . $name;
            $callable = array($this, $method);
            if (is_callable($callable)) {
                call_user_func($callable, $value);
            }
        }
        return $this;
    }

    /**
     * Get request data and return it in a big array
     * @return type
     */
    public function getData()
    {
        $data['serverSideInfo']     = $this->getServerSideInfo();
        $data['clientSideInfo']     = $this->getClientSideInfo();
        $data['environmentInfo']    = $this->getEnvironmentInfo();
        $data['customData']         = $this->getCustomData();
        $data['detectionProperties']= $this->getDetectionProperties();

        return $data;
    }


    /**
     * Set server variables to Request
     * @param   array   $_SERVER
     * @return  \Core_Detector_Request
     */
    public function setServerSideInfo($server)
    {
        $this->_serverSideInfo  = $server;
        return $this;
    }

    /**
     * Get server settings from request, or if empty, return $_SERVER array
     * @return  array
     */
    public function getServerSideInfo()
    {
        if (empty($this->_serverSideInfo)) {
            return $_SERVER;
        }
        return $this->_serverSideInfo;
    }

    /**
     * Set custom data to request ( key => val )
     * @param   array   $customData
     * @return  \Core_Detector_Request
     */
    public function setCustomData($customData)
    {
        $this->_customData  = array_merge($this->_customData, $customData);
        return $this;
    }

    /**
     * Get custom data from request
     * @return  array
     */
    public function getCustomData()
    {
        return $this->_customData;
    }


    /**
     * Set POST data to request
     * @param   array   $post
     * @return  \Core_Detector_Request
     */
    public function setClientSideInfo($clientSideInfo)
    {
        $this->_clientSideInfo  = $clientSideInfo;
        return $this;
    }

    /**
     * Get post parameters from request, if not set, get from $_POST directly
     * @return  array
     */
    public function getClientSideInfo()
    {
        if (empty($this->_clientSideInfo)) {
            return $_POST;
        }
        return $this->_clientSideInfo;
    }


    /**
     * Sets application environment info to Request object
     * @param   array   $environmentInfo
     * @return  \Core_Detector_Request
     */
    public function setEnvironmentInfo($envirnomentInfo)
    {
        $this->_environmentInfo  = $envirnomentInfo;
        return $this;
    }

    /**
     * Returns environment info
     * @return  array
     */

    public function getEnvironmentInfo()
    {
        return $this->_environmentInfo;
    }


    /**
     * Set detection properties we want to have returned in the response
     * @param   array   $detectionProperties
     * @return  \Core_Detector_Request
     */
    public function setDetectionProperties($detectionProperties)
    {
        // merge default return value with additional return values we need, and make sure all of them are unique
        $this->_detectionProperties = array_unique(array_merge($detectionProperties, $this->_detectionProperties));
        return $this;
    }

    /**
     * Get detection properties
     * @return  array
     */
    public function getDetectionProperties()
    {
        if (empty($this->_detectionProperties)) {
            return array('detection_id');
        }
        return $this->_detectionProperties;
    }

}